import { Component, Input, Output, EventEmitter, Inject } from '@angular/core';
import { FormGroup, FormControl } from '@angular/forms';
import { KeyboardUtil } from '../../utils/keyboard-util';
import { DOCUMENT } from '@angular/common';
import * as i0 from "@angular/core";
import * as i1 from "../../pipes/keys.pipe";
import * as i2 from "@angular/common";
import * as i3 from "@angular/forms";
export class NgOtpInputComponent {
    constructor(keysPipe, document) {
        this.keysPipe = keysPipe;
        this.document = document;
        this.config = { length: 4 };
        // tslint:disable-next-line: no-output-on-prefix
        this.onInputChange = new EventEmitter();
        this.inputControls = new Array(this.config.length);
        this.componentKey = Math.random()
            .toString(36)
            .substring(2) + new Date().getTime().toString(36);
    }
    get inputType() {
        return this.config?.isPasswordInput
            ? 'password'
            : this.config?.allowNumbersOnly
                ? 'tel'
                : 'text';
    }
    ngOnInit() {
        this.otpForm = new FormGroup({});
        for (let index = 0; index < this.config.length; index++) {
            this.otpForm.addControl(this.getControlName(index), new FormControl());
        }
        this.otpForm.valueChanges.subscribe((v) => {
            this.keysPipe.transform(this.otpForm.controls).forEach((k) => {
                var val = this.otpForm.controls[k].value;
                if (val && val.length > 1) {
                    if (val.length >= this.config.length) {
                        this.setValue(val);
                    }
                    else {
                        this.rebuildValue();
                    }
                }
            });
        });
    }
    ngAfterViewInit() {
        if (!this.config.disableAutoFocus) {
            const containerItem = this.document.getElementById(`c_${this.componentKey}`);
            if (containerItem) {
                const ele = containerItem.getElementsByClassName('otp-input')[0];
                if (ele && ele.focus) {
                    ele.focus();
                }
            }
        }
    }
    getControlName(idx) {
        return `ctrl_${idx}`;
    }
    onKeyDown($event, inputIdx) {
        const prevInputId = this.getBoxId(inputIdx - 1);
        const currentInputId = this.getBoxId(inputIdx);
        if (KeyboardUtil.ifSpacebar($event)) {
            $event.preventDefault();
            return false;
        }
        if (KeyboardUtil.ifBackspace($event)) {
            if (!$event.target.value) {
                this.clearInput(prevInputId, inputIdx - 1);
                this.setSelected(prevInputId);
            }
            else {
                this.clearInput(currentInputId, inputIdx);
            }
            this.rebuildValue();
            return;
        }
    }
    onInput($event) {
        let newVal = this.currentVal ? `${this.currentVal}${$event.target.value}` : $event.target.value;
        if (this.config.allowNumbersOnly && !this.validateNumber(newVal)) {
            $event.target.value = '';
            $event.stopPropagation();
            $event.preventDefault();
            return;
        }
    }
    onKeyUp($event, inputIdx) {
        if (KeyboardUtil.ifTab($event)) {
            inputIdx -= 1;
        }
        const nextInputId = this.getBoxId(inputIdx + 1);
        const prevInputId = this.getBoxId(inputIdx - 1);
        const currentInputId = this.getBoxId(inputIdx);
        if (KeyboardUtil.ifRightArrow($event)) {
            $event.preventDefault();
            this.setSelected(nextInputId);
            return;
        }
        if (KeyboardUtil.ifLeftArrow($event)) {
            $event.preventDefault();
            this.setSelected(prevInputId);
            return;
        }
        if (KeyboardUtil.ifDelete($event)) {
            if (!$event.target.value) {
                this.clearInput(prevInputId, inputIdx - 1);
                this.setSelected(prevInputId);
            }
            else {
                this.clearInput(currentInputId, inputIdx);
            }
            this.rebuildValue();
            return;
        }
        if (!$event.target.value) {
            return;
        }
        if (this.ifValidKeyCode($event)) {
            this.setSelected(nextInputId);
        }
        this.rebuildValue();
    }
    validateNumber(val) {
        return val && /^[0-9]+$/.test(val);
    }
    getBoxId(idx) {
        return `otp_${idx}_${this.componentKey}`;
    }
    clearInput(eleId, inputIdx) {
        let ctrlName = this.getControlName(inputIdx);
        this.otpForm.controls[ctrlName]?.setValue(null);
        const ele = this.document.getElementById(eleId);
        if (ele && ele instanceof HTMLInputElement) {
            ele.value = null;
        }
    }
    setSelected(eleId) {
        this.focusTo(eleId);
        const ele = this.document.getElementById(eleId);
        if (ele && ele.setSelectionRange) {
            setTimeout(() => {
                ele.setSelectionRange(0, 1);
            }, 0);
        }
    }
    ifValidKeyCode(event) {
        const inp = event.key;
        const isMobile = /iPhone|iPad|iPod|Android/i.test(navigator.userAgent);
        return (isMobile ||
            /[a-zA-Z0-9-_]/.test(inp));
    }
    focusTo(eleId) {
        const ele = this.document.getElementById(eleId);
        if (ele) {
            ele.focus();
        }
    }
    // method to set component value
    setValue(value) {
        if (this.config.allowNumbersOnly && isNaN(value)) {
            return;
        }
        this.otpForm.reset();
        if (!value) {
            this.rebuildValue();
            return;
        }
        value = value.toString().replace(/\s/g, ''); // remove whitespace
        Array.from(value).forEach((c, idx) => {
            if (this.otpForm.get(this.getControlName(idx))) {
                this.otpForm.get(this.getControlName(idx)).setValue(c);
            }
        });
        if (!this.config.disableAutoFocus) {
            const containerItem = this.document.getElementById(`c_${this.componentKey}`);
            var indexOfElementToFocus = value.length < this.config.length ? value.length : (this.config.length - 1);
            let ele = containerItem.getElementsByClassName('otp-input')[indexOfElementToFocus];
            if (ele && ele.focus) {
                ele.focus();
            }
        }
        this.rebuildValue();
    }
    rebuildValue() {
        let val = '';
        this.keysPipe.transform(this.otpForm.controls).forEach(k => {
            if (this.otpForm.controls[k].value) {
                let ctrlVal = this.otpForm.controls[k].value;
                let isLengthExceed = ctrlVal.length > 1;
                let isCaseTransformEnabled = !this.config.allowNumbersOnly && this.config.letterCase && (this.config.letterCase.toLocaleLowerCase() == 'upper' || this.config.letterCase.toLocaleLowerCase() == 'lower');
                ctrlVal = ctrlVal[0];
                let transformedVal = isCaseTransformEnabled ? this.config.letterCase.toLocaleLowerCase() == 'upper' ? ctrlVal.toUpperCase() : ctrlVal.toLowerCase() : ctrlVal;
                if (isCaseTransformEnabled && transformedVal == ctrlVal) {
                    isCaseTransformEnabled = false;
                }
                else {
                    ctrlVal = transformedVal;
                }
                val += ctrlVal;
                if (isLengthExceed || isCaseTransformEnabled) {
                    this.otpForm.controls[k].setValue(ctrlVal);
                }
            }
        });
        if (this.formCtrl?.setValue) {
            this.formCtrl.setValue(val);
        }
        this.onInputChange.emit(val);
        this.currentVal = val;
    }
    handlePaste(e) {
        // Get pasted data via clipboard API
        let clipboardData = e.clipboardData || window['clipboardData'];
        if (clipboardData) {
            var pastedData = clipboardData.getData('Text');
        }
        // Stop data actually being pasted into div
        e.stopPropagation();
        e.preventDefault();
        if (!pastedData || (this.config.allowNumbersOnly && !this.validateNumber(pastedData))) {
            return;
        }
        this.setValue(pastedData);
    }
}
/** @nocollapse */ NgOtpInputComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.3.12", ngImport: i0, type: NgOtpInputComponent, deps: [{ token: i1.KeysPipe }, { token: DOCUMENT }], target: i0.ɵɵFactoryTarget.Component });
/** @nocollapse */ NgOtpInputComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "13.3.12", type: NgOtpInputComponent, selector: "ng-otp-input", inputs: { config: "config", formCtrl: "formCtrl" }, outputs: { onInputChange: "onInputChange" }, ngImport: i0, template: "<div class=\"ng-otp-input-wrapper wrapper {{config.containerClass}}\" id=\"c_{{componentKey}}\" *ngIf=\"otpForm?.controls\"\r\n  [ngStyle]=\"config.containerStyles\">\r\n  <input (paste)=\"handlePaste($event)\" [pattern]=\"config.allowNumbersOnly ? '\\\\d*' : ''\" [type]=\"inputType\"  [placeholder]=\"config?.placeholder || ''\"\r\n    [ngStyle]=\"config.inputStyles\" \r\n    class=\"otp-input {{config.inputClass}}\" autocomplete=\"one-time-code\" *ngFor=\"let item of otpForm?.controls | keys;let i=index\"\r\n    [formControl]=\"otpForm.controls[item]\" #inp [id]=\"getBoxId(i)\" \r\n    (keyup)=\"onKeyUp($event,i)\" (input)=\"onInput($event)\" (keydown)=\"onKeyDown($event,i)\" >\r\n</div>", styles: [".otp-input{width:50px;height:50px;border-radius:4px;border:solid 1px #c5c5c5;text-align:center;font-size:32px}.ng-otp-input-wrapper .otp-input:not(:last-child){margin-right:8px}@media screen and (max-width: 767px){.otp-input{width:40px;font-size:24px;height:40px}}@media screen and (max-width: 420px){.otp-input{width:30px;font-size:18px;height:30px}}\n"], directives: [{ type: i2.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: i2.NgStyle, selector: "[ngStyle]", inputs: ["ngStyle"] }, { type: i2.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { type: i3.DefaultValueAccessor, selector: "input:not([type=checkbox])[formControlName],textarea[formControlName],input:not([type=checkbox])[formControl],textarea[formControl],input:not([type=checkbox])[ngModel],textarea[ngModel],[ngDefaultControl]" }, { type: i3.PatternValidator, selector: "[pattern][formControlName],[pattern][formControl],[pattern][ngModel]", inputs: ["pattern"] }, { type: i3.NgControlStatus, selector: "[formControlName],[ngModel],[formControl]" }, { type: i3.FormControlDirective, selector: "[formControl]", inputs: ["formControl", "disabled", "ngModel"], outputs: ["ngModelChange"], exportAs: ["ngForm"] }], pipes: { "keys": i1.KeysPipe } });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.3.12", ngImport: i0, type: NgOtpInputComponent, decorators: [{
            type: Component,
            args: [{ selector: 'ng-otp-input', template: "<div class=\"ng-otp-input-wrapper wrapper {{config.containerClass}}\" id=\"c_{{componentKey}}\" *ngIf=\"otpForm?.controls\"\r\n  [ngStyle]=\"config.containerStyles\">\r\n  <input (paste)=\"handlePaste($event)\" [pattern]=\"config.allowNumbersOnly ? '\\\\d*' : ''\" [type]=\"inputType\"  [placeholder]=\"config?.placeholder || ''\"\r\n    [ngStyle]=\"config.inputStyles\" \r\n    class=\"otp-input {{config.inputClass}}\" autocomplete=\"one-time-code\" *ngFor=\"let item of otpForm?.controls | keys;let i=index\"\r\n    [formControl]=\"otpForm.controls[item]\" #inp [id]=\"getBoxId(i)\" \r\n    (keyup)=\"onKeyUp($event,i)\" (input)=\"onInput($event)\" (keydown)=\"onKeyDown($event,i)\" >\r\n</div>", styles: [".otp-input{width:50px;height:50px;border-radius:4px;border:solid 1px #c5c5c5;text-align:center;font-size:32px}.ng-otp-input-wrapper .otp-input:not(:last-child){margin-right:8px}@media screen and (max-width: 767px){.otp-input{width:40px;font-size:24px;height:40px}}@media screen and (max-width: 420px){.otp-input{width:30px;font-size:18px;height:30px}}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.KeysPipe }, { type: Document, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }]; }, propDecorators: { config: [{
                type: Input
            }], onInputChange: [{
                type: Output
            }], formCtrl: [{
                type: Input
            }] } });
//# sourceMappingURL=data:application/json;base64,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